#include "process.h"
#include "frame.h"
#include "symtab.h"
#include "symbol.h"
#include "core.h"
#include "sunossymtab.h"
#ifndef SUN3PI
#include "coff.h"
#include "ieeesymtab.h"
#endif	/* SUN3PI */
#include "asm.h"
#include "m68kasm.h"
#include "m68kcore.h"
SRCFILE("m68kcore.c")

const short M68K_RTS = 0x4E75;
const short M68K_BSR = 0x6100, M68K_JSR = 0x4e80;
const short M68K_BSR_MSK = 0xff00, M68K_JSR_MSK = 0xffc0;
const short LINKA6=0x4E56, ADDLSP=0xDFFC;
const short MOVEML_MSK=0xFFC0, MOVEML=0x48C0;
const short MOVEMLSP=0x48D7, MOVEMLSPDEC=0x48E7, MOVEMLFP=0x48EE;
const short MOVLD0FPREL=0x2D40;

int M68kCore::REG_AP()			{ return 14; }
int M68kCore::REG_FP()			{ return 14; }
int M68kCore::REG_SP()			{ return 15; }
int M68kCore::REG_PC()			{ return 17; }
int M68kCore::nregs()			{ return 18; }
long M68kCore::returnregloc()		{ return regloc(0); }
long M68kCore::callingpc(long fp)	{ return peek(fp+4)->lng; }
long M68kCore::callingfp(long fp)	{ return peek(fp)->lng; }
int M68kCore::atreturn(long pc)		{ return peek(pc)->sht == M68K_RTS; }
int M68kCore::argstart()		{ return 8; }
Asm *M68kCore::newAsm()			{ return new M68kAsm(this); }

void M68kCore::newSymTab(long reloc)
{
#ifndef SUN3PI
	short magic = 0;
	lseek(stabfd, 0L, 0);
	ReadOK(stabfd, (char*)&magic, sizeof(magic));
	if (magic == 0520)
		 _symtab = new CoffSymTab(this, stabfd, _symtab, reloc);
	else if (*(unsigned char *)&magic == 0xE0)
		 _symtab = new IeeeSymTab(this, stabfd, _symtab, reloc);
	else
#endif	/* SUN3PI */
		 _symtab = new SunosSymTab(this, stabfd, _symtab, reloc);
}

M68kCore::M68kCore()
{
	stackdir = GROWDOWN;
	memlayout = MSBFIRST;
	bptsize = 2;
}

char *M68kCore::regname(int r)
{
	static char *regnames[] = {
		"d0", "d1", "d2", "d3", "d4", "d5", "d6", "d7",
		"a0", "a1", "a2", "a3", "a4", "a5", "fp", "sp",
		"ps", "pc", "usp", "isp", "vbr", "sfc", "dfc", "msp",
		"cacr", "caar"
	};
	if (r < nregs())
		return regnames[r];
	else
		return 0;
}

#define REGBIT(r) (1 << r)
long M68kCore::saved(Frame *f, int r, int sz)
{
	if( r > 15 )
		return 0;
	if( !(f->regsave & REGBIT(r)) )
		return 0;
	long loc = f->regbase;
	while( --r >= 0 )
		if( f->regsave & REGBIT(r) ) loc += 4;
	return loc + 4 - sz;
}

Frame M68kCore::frameabove(long _fp)
{
	int i, j;
	Frame f(this);
	if( _fp ){
		f.pc = callingpc(_fp);
		f.fp = callingfp(_fp);
	} else {
		f.pc = pc();
		f.fp = fp();
	}
	f.ap = f.fp;
	f.regbase = f.fp;
	Func *funcp = (Func*)_symtab->loctosym(U_FUNC, f.pc);
	if (funcp) {
		int faddr = (int)funcp->range.lo;
		short instr = peek(faddr)->sht;
		if (instr == LINKA6) {
			f.regbase += peek(faddr+2)->sht;
			faddr += 4;
			instr = peek(faddr)->sht;
		}
		if (instr == ADDLSP) {
			f.regbase += peek(faddr+2)->lng;
			faddr += 6;
			instr = peek(faddr)->sht;
		}
		if (instr == MOVEMLSP)		// Sun compiler
			f.regsave = peek(faddr+2)->sht;
		else if (instr == MOVEMLFP) {	// mcc compiler
			f.regsave = peek(faddr+2)->sht;
			f.regbase = f.fp + peek(faddr+4)->sht;
		}
		else if (instr == MOVEMLSPDEC) { // m68ksgs compiler
			int mask = peek(faddr+2)->sht & 0xFFFF;
			for(j = 0, i = 0; i < 16; i++) {
				j <<= 1;
				if (mask & 1) {
					f.regbase -= 4;
					j |= 1;
				}
				mask >>= 1;
			}
			f.regsave = j;
		}
	}
	return f;
}

long M68kCore::instrafterjsr()
{
	dostep(0,0,1);
	return peek(sp())->lng;
}

int M68kCore::atjsr(long pc)
{
	short inst = peek(pc)->sht;
	return (inst & M68K_BSR_MSK) == M68K_BSR ||
	       (inst & M68K_JSR_MSK) == M68K_JSR;
}

char *M68kCore::stepprolog()
{
	Func *f = (Func*)_symtab->loctosym(U_FUNC, pc());
	Stmt *s;
	if (f && (s = f->stmt(pc())) && s->range.lo == f->range.lo)
		process()->stmtstep(1);
	else {
		short inst = peek(pc())->sht;
		if (inst == LINKA6) {
			dostep(0,0,1);
			inst = peek(pc())->sht;
		}
		if (inst == ADDLSP) {	// sun 68k compiler
			dostep(0,0,1);
			inst = peek(pc())->sht;
		}
		if ((inst & MOVEML_MSK) == MOVEML) {
			dostep(0,0,1);
			inst = peek(pc())->sht;
		}
		if (inst == MOVLD0FPREL) //m68ksgs function that returns struct
			dostep(0,0,1);
	}
	return 0;
}

char *M68kCore::docall(long addr, int)
{
	const int CALL_SIZE=6, JSR=0x4eb9;
	char save[CALL_SIZE], *error;

	if( behavetype() == ACTIVE )
		return "process not stopped";
	int callstart = (int)scratchaddr();
	if( ( error = read(callstart, save, CALL_SIZE) )
	 || ( error = poke(callstart+0, JSR, 2) )
	 || ( error = poke(callstart+2, addr, 4) )
	 || ( error = regpoke(REG_PC(), callstart) )
	 || ( error = step(callstart, callstart+CALL_SIZE) )
	 || ( error = write(callstart, save, CALL_SIZE) ) )
		return error;
	return 0;
}

long M68kCore::apforcall(int argbytes)
{
	regpoke(REG_SP(), sp() - argbytes);
	return sp() - 8;
}
